## Pacotes
library(xtable)
library(bbmle)
library(MASS)
library(sads)


################################################################################
## Exemplo de modelo binomial
################################################################################
## Gera os dados
set.seed(42)
dist <- rep(c(0,0.5,1,2,4,10,20,50),each=3)
p.esp <- exp(-3+0.05*dist)/(1+exp(-3+0.05*dist))
nobs <- rbinom(24,size=100,prob=p.esp)
pred.seed <- data.frame(distancia=dist, n.sobrev=nobs)

## ----dados-seeds-summary, tidy=FALSE------------------------------------------
head(pred.seed)


## ----grafico-seeds, echo=FALSE, fig=TRUE, out.width="0.6\\textwidth"----------
plot(I(n.sobrev/100)~dist, data=pred.seed, 
     xlab="Distância da árvore-mãe", ylab="Proporção sobrevivente")


## ----logistica-seed-----------------------------------------------------------
logistica <- function(X, a0, a1){
    exp(a0 + a1*X) / (1 + exp(a0 + a1*X))
}


## ----log-lik-seed-------------------------------------------------------------
seed.LL1 <- function(a0, a1){
    eta  <- logistica(X=pred.seed$distancia, 
                              a0=a0, a1=a1)
    -sum(dbinom(pred.seed$n.sobrev, size = 100, 
                p = eta, log=T))
}


## ----binomial-starting-values-------------------------------------------------
p1 <- pred.seed$n.sobrev/100
logito1 <- log( p1 / (1-p1) )
lm1 <- lm(logito1~pred.seed$distancia)
(cf1 <- coef(lm1))


## ----Ajuste-numérico----------------------------------------------------------
seed.m1 <- mle2(seed.LL1, 
                start=list(a0=cf1[1], a1=cf1[2]) )


## ----seeds-coefs, include=FALSE-----------------------------------------------
(cf2 <- coef(seed.m1))


## ----seeds-predicted-plot-code, echo=FALSE, include=FALSE---------------------
plot(I(n.sobrev/100)~dist, data=pred.seed, 
     xlab="Distância da árvore-mãe", 
     ylab="Proporção sobrevivente")
curve(logistica(x, a0 = cf2[1], a1 = cf2[2]), 
      add=T, col="blue")


## ----seeds-coefs, echo=TRUE---------------------------------------------------
(cf2 <- coef(seed.m1))


## ----seeds-predicted-plot-code, echo=FALSE, fig=TRUE, out.width="0.6\\textwidth"----
plot(I(n.sobrev/100)~dist, data=pred.seed, 
     xlab="Distância da árvore-mãe", 
     ylab="Proporção sobrevivente")
curve(logistica(x, a0 = cf2[1], a1 = cf2[2]), 
      add=T, col="blue")


## ----seeds-profile,echo=F, fig=T, fig.width=12, fig.height=6------------------
par(mfrow = c(1,2))
plotprofmle( profile(seed.m1) )
par( mfrow=c(1,1) )


## ----seeds-intervals----------------------------------------------------------
likelregions( profile(seed.m1) )


## ----seeds-ci-----------------------------------------------------------------
confint( seed.m1 )


## ----log-lik-seed-m0----------------------------------------------------------
seed.LL0 <- function(a0){
    eta <- exp(a0)/(1+exp(a0))
    -sum(dbinom(pred.seed$n.sobrev,
                size=100, p=eta, log=T)) }


## ----log-lik-seed-simple------------------------------------------------------
seed.LL0 <- function(p){
    -sum(dbinom(pred.seed$n.sobrev,size=100,
                prob=p, log=T)) }


## ----seeds-model0-------------------------------------------------------------
seed.m0 <- mle2(seed.LL0, 
                start=list(
                    p=sum(pred.seed$n.sobrev)/2400)
                )


## ----seeds-aic----------------------------------------------------------------
AICtab(seed.m0, seed.m1, base=T)


## ----compara-modelos,echo=F, fig=T, out.width="0.6\\textwidth"----------------
plot(I(n.sobrev/100)~dist, data=pred.seed, 
     xlab="Distância da árvore-mãe", 
     ylab="Proporção sobrevivente")
curve(logistica(x, a0 = cf2[1], a1 = cf2[2]), add=T, col="blue")
abline(h=coef(seed.m0), lty=2,col="red")


################################################################################
## Exemplo de modelo Poisson
################################################################################

## Gera dados para o exemplo
## ----aves-data, echo=F--------------------------------------------------------
set.seed(42)
area <- round(runif(20,50,800))
ncap <- rpois(20,lambda=exp(-5+1*log(area)))
aves <- data.frame(area=area,ncap=ncap)


## ----aves-data-summary, tidy=FALSE--------------------------------------------
head(aves)


## ----scatterplot-poisson, fig=T, echo=F, out.width="0.6\\textwidth"-----------
plot(ncap~log(area), data=aves,
     xlab="Ln área do fragmento (ha)",
     ylab="N Capturas")


## ----log-lik-aves-------------------------------------------------------------
aves.LL1 <- function(a0, a1) {
  eta <-  exp( a0 + a1*log(aves$area) )
  -sum( dpois(aves$ncap, lambda = eta, log = T) )
}


## ----poisson-starting-values--------------------------------------------------
pm1 <- lm(log(ncap+0.1)~log(aves$area))
( cf1 <- coef(pm1) )


## ----aves-ajuste-numérico-----------------------------------------------------
aves.m1 <- mle2(aves.LL1,
                start= list(a0 = cf1[1],
                            a1 = cf1[2]) )


## ----aves-coefs---------------------------------------------------------------
( cf3 <- coef(aves.m1) )


## ----aves-predicted-plot-code, include=FALSE, echo=FALSE----------------------
f3 <- function(x) exp(cf3[1]+cf3[2]*x)

plot(ncap~log(area), data = aves,
     xlab = "Ln área do fragmento (ha)",
     ylab = "N de Capturas")
curve(f3(x), add = T, col = "blue")


## ----aves-predicted-plot-code, fig=T, echo=FALSE, out.width="0.6\\textwidth"----
f3 <- function(x) exp(cf3[1]+cf3[2]*x)

plot(ncap~log(area), data = aves,
     xlab = "Ln área do fragmento (ha)",
     ylab = "N de Capturas")
curve(f3(x), add = T, col = "blue")


## ----aves-profile,echo=F, fig=T, fig.width=12, fig.height=6-------------------
par(mfrow=c(1,2))
plotprofmle(profile(aves.m1))
par(mfrow=c(1,1))


## ----aves-intervals-----------------------------------------------------------
likelregions( profile(aves.m1) )


## ----aves-ci------------------------------------------------------------------
confint( aves.m1 )


## ----log-lik-aves-m0----------------------------------------------------------
aves.LL0 <- function(a0){
    eta <- exp(a0)
    -sum(dpois(aves$ncap, lambda=eta, log=T))
}


## ----aves-model0--------------------------------------------------------------
aves.m0 <- mle2(aves.LL0, 
                start=list(
                    a0=log(mean(aves$ncap))))


## ----aves-aic-----------------------------------------------------------------
AICtab(aves.m0,aves.m1, base=T)


## ----compara-modelos-aves,echo=F, fig=T, out.width="0.6\\textwidth"-----------
plot(ncap~log(area), 
     data=aves,
     xlab="Ln área do fragmento (ha)",
     ylab="N de Capturas")
curve(f3(x),add=T, col="blue")
abline(h=exp(coef(aves.m0)), lty=2,col="red")


################################################################################
## GLMs
################################################################################

## ----glm-aves-----------------------------------------------------------------
aves.glm1 <- glm(ncap~log(area), family=poisson)
coef(aves.glm1)
coef(aves.m1)


## ----glm-aves-logLik----------------------------------------------------------
logLik(aves.glm1)
logLik(aves.m1)


## ----residuos-glm-aves, echo=F, fig=T, out.width="0.6\\textwidth"-------------
par(mfrow=c(2,2))
plot(aves.glm1)
par(mfrow=c(1,1))


## ----iwls-prep, echo=F--------------------------------------------------------
## Exemplo de ajuste de glm com IWLS
## Exemplo 4.4 da Dodson: glm com link=indentity
## Os pesos sao as variancias previstas pelo modelo para cada ponto
set.seed(22)
x <- runif(20,0,6)
y <- rpois(20,lambda=1+0.5*x)
## Regressão linear
m0 <- lm(y~x)
## Regressao linear ponderada pelas variancias estimadas pela anterior
m1 <- lm(y~x, weights=1/fitted(m0))
## Segunda iteracao
m2 <- lm(y~x, weights=1/fitted(m1))
## Terceira iteracao
m3 <- lm(y~x, weights=1/fitted(m2))
## Quarta
m4 <- lm(y~x, weights=1/fitted(m3))
## Quinta
m5 <- lm(y~x, weights=1/fitted(m4))
## Sexta
m6 <- lm(y~x, weights=1/fitted(m5))
## Funcao para plotar
plot.peso <- function(x,y,m0,m1,cor.peso=0.05,...){
  plot(y~x, cex=cor.peso/(fitted(m0)/sum(fitted(m0))),...)
  abline(m0, lty=2)
  abline(m1, col="blue")
}

## Os plots
## ----iwls-1, fig=T, echo=F, out.width="0.6\\textwidth"------------------------
plot.peso(x,y, m0,m1, xlim=c(0,6), ylim=c(0,6))


## ----iwls-2, fig=T, echo=F, out.width="0.6\\textwidth"------------------------
plot.peso(x,y, m1,m2, xlim=c(0,6), ylim=c(0,6))


## ----iwls-3, fig=T, echo=F, out.width="0.6\\textwidth"------------------------
plot.peso(x,y, m2,m3, xlim=c(0,6), ylim=c(0,6))

